<?php

namespace Modules\Academic\Http\Controllers\API;

use Exception;
use App\Traits\Trackable;
use Illuminate\Http\Request;
use App\Imports\StudentsImport;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Hash;
use Maatwebsite\Excel\Facades\Excel;
use Modules\Academic\Models\Student;
use App\Traits\StudentCollectionTrait;
use Modules\Authentication\Models\User;
use Illuminate\Support\Facades\Validator;
use Modules\Academic\Models\LibraryMember;
use Modules\Academic\Services\UserService;
use Modules\Academic\Models\StudentSession;
use Modules\Academic\Services\StudentService;
use Modules\Academic\Services\StudentSessionService;
use Modules\Academic\Http\Requests\StudentCreateRequest;
use Modules\Academic\Http\Requests\StudentUpdateRequest;

class APIStudentController extends Controller
{
    use StudentCollectionTrait, Trackable;

    public function __construct(
        private readonly UserService $userService,
        private readonly StudentService $studentService,
        private readonly StudentSessionService $studentSessionService
    ) {}

    public function index(Request $request): JsonResponse
    {
        $class_id = (int) request()->class_id;
        $section_id = (int) request()->section_id;
        $group_id = (int) request()->group_id;
        $perPage = (int) request()->per_page;

        $students = $this->studentService->getStudentsByClassSectionGroup($class_id, $section_id, $group_id, $perPage,  get_institute_id(), get_branch_id());
        if (! $students) {
            return $this->responseError([], _lang('Students not found.'), 404);
        }

        return $this->responseSuccess($students, 'Students have been fetched successfully.');
    }

    public function store(StudentCreateRequest $request): JsonResponse
    {
        try {
            DB::beginTransaction();
            $password = '123456';

            if ($request->hasFile('image') && $request->file('image')->isValid()) {
                $imagePath = fileUploader('users/', 'png', $request->file('image'));
            }
            $user = $this->userService->createOrUpdateUser([
                'name' => $request->first_name,
                'email' => $request->email, // Fixed issue, previously set as roll
                'phone' => $request->phone,
                'password' => $request->password ?? $password,
                'role_id' => 4,
                'status' => 1,
                'image' => $imagePath,
                'user_type' => 'Student',
            ]);

            $student = $this->studentService->createStudent([
                'user_id' => $user->id,
                'first_name' => $request->first_name,
                'last_name' => $request->last_name ?? null,
                'father_name' => $request->father_name ?? null,
                'mother_name' => $request->mother_name ?? null,
                'birthday' => $request->birthday ?? null,
                'gender' => $request->gender ?? null,
                'blood_group' => $request->blood_group ?? null,
                'religion' => $request->religion ?? null,
                'phone' => $request->phone,
                'register_no' => intval($request->register_no ?? 0),
                'roll' => intval($request->roll),
                'address' => $request->address ?? null,
                'group' => $request->group ?? null,
                'access_key' => $request->password ?? $password,
                'information_sent_to_name' => $request->information_sent_to_name ?? null,
                'information_sent_to_relation' => $request->information_sent_to_relation ?? null,
                'information_sent_to_phone' => $request->information_sent_to_phone ?? null,
                'information_sent_to_address' => $request->information_sent_to_address ?? null,
            ], $user->id);

            $sessionData = [
                'session_id' => get_option('academic_year'),
                'student_id' => intval($student->id),
                'class_id' => intval($request->class_id),
                'section_id' => intval($request->section_id),
                'roll' => intval($request->roll),
                'qr_code' => str_pad(random_int(0, 9999999999999999), 16, '0', STR_PAD_LEFT),
            ];
            $this->studentSessionService->createStudentSession($sessionData);

            // Library Membership
            LibraryMember::create([
                'institute_id' => get_institute_id(),
                'branch_id' => get_branch_id(),
                'user_id' => $user->id,
                'member_type' => 'Student',
                'library_id' => intval($request->roll),
                'student_id' => $student->id,
            ]);

            DB::commit();

            return $this->responseSuccess($student, 'Student has been created successfully.');
        } catch (Exception $e) {
            DB::rollback();

            return $this->responseError([], $e->getMessage());
        }
    }

    public function show(int $id): JsonResponse
    {
        $student = $this->studentService->findStudentById((int) $id);
        if (! $student) {
            return $this->responseError([], _lang('Student not found.'), 404);
        }

        return $this->responseSuccess($student, 'Student have been fetched successfully.');
    }

    public function update(StudentUpdateRequest $request, $id): JsonResponse
    {
        $student = $this->studentService->findStudentById((int) $id);
        if (! $student) {
            return $this->responseError([], _lang('Student not found.'), 404);
        }

        try {
            DB::beginTransaction();
            $password = '123456';

            if (! empty($request['image']) && $request['image'] instanceof \Illuminate\Http\UploadedFile) {
                $imagePath = fileUploader('users/', 'png', $request['image'], $student->user?->image);
            }
            $user = $this->userService->createOrUpdateUser([
                'name' => $request->first_name,
                'email' => $request->email,
                'phone' => $request->phone,
                'password' => $request->password ?? $password,
                'user_type' => 'Student',
                'role_id' => 4, // ✅ FIXED: Ensure role_id is included
                'status' => 1, // Ensure user is active
                'image' =>  $imagePath,
            ], (int) $student->user_id);

            // ✅ Update Student information.
            $this->studentService->updateStudent([
                'user_id' => $user->id,
                'first_name' => $request->first_name,
                'last_name' => $request->last_name,
                'father_name' => $request->father_name,
                'mother_name' => $request->mother_name,
                'birthday' => $request->birthday,
                'gender' => $request->gender,
                'blood_group' => $request->blood_group,
                'religion' => $request->religion,
                'phone' => $request->phone,
                'register_no' => intval($request->register_no ?? 0),
                'roll' => intval($request->roll),
                'address' => $request->address,
                'group' => $request->group,
                'access_key' => $request->password ?? $password,
            ], (int) $student->id);

            // ✅ Update or Create Library Membership
            LibraryMember::updateOrCreate(
                ['user_id' => $user->id, 'student_id' => $student->id],
                ['library_id' => intval($request->roll)]
            );

            // ✅ Update Student Session
            $studentSession = $this->studentSessionService->findStudentSessionByStudentId((int) $student->id);
            $sessionData = [
                'class_id' => intval($request->class_id),
                'section_id' => intval($request->section_id),
                'roll' => intval($request->roll),
                'qr_code' => str_pad(random_int(0, 9999999999999999), 16, '0', STR_PAD_LEFT),
            ];
            if ($studentSession) {
                $this->studentSessionService->updateStudentSession($sessionData, $studentSession->id);
            }

            DB::commit();

            $student = $this->studentService->findStudentById((int) $id);
            return $this->responseSuccess($student, 'Student has been updated successfully.');
        } catch (Exception $e) {
            DB::rollback();

            return $this->responseError([], $e->getMessage());
        }
    }

    public function destroy(int $id): JsonResponse
    {
        try {
            return DB::transaction(function () use ($id) {
                $student = $this->studentService->findStudentById(intval($id));

                // Check if student exists
                if (! $student) {
                    throw new Exception('Student not found.');
                }

                // Delete student session only if it exists
                if (! empty($student->studentSession)) {
                    $this->studentSessionService->deleteStudentSessionById($student->studentSession->id);
                }

                // Delete user if exists
                if (! empty($student->user_id)) {
                    $this->userService->deleteUserById(intval($student->user_id));
                }

                // Delete student record
                $this->studentService->deleteStudentById(intval($student->id));

                return $this->responseSuccess([], 'Student has been deleted successfully.');
            });
        } catch (Exception $e) {
            return $this->responseError([], $e->getMessage());
        }
    }

    public function updateStatus(Request $request): JsonResponse
    {
        $request->validate([
            'student_id' => 'required|integer',
            'type' => 'required|string|in:enable,disable', // Accepts only "enable" or "disable"
        ]);

        $student = $this->studentService->findStudentById((int) $request->student_id);
        if (! $student) {
            return $this->responseError([], 'Student Not Found', 404);
        }

        // Determine the status based on the type
        $status = $request->type === 'enable' ? '1' : '0';

        // Update student status
        $student->update(['status' => $status]);

        // Update associated user status
        $user = $this->userService->findUserById((int) $student->user_id);
        if ($user) {
            $user->update(['user_status' => $status]);
        }

        $message = $request->type === 'enable'
            ? 'Student has been enabled successfully.'
            : 'Student has been disabled successfully.';

        return $this->responseSuccess([], $message);
    }

    public function multipleDelete(Request $request): JsonResponse
    {
        $request->validate([
            'student_ids' => 'required|array|min:1',
            'student_ids.*' => 'integer|exists:students,id',
        ]);

        foreach ($request->student_ids as $key => $student_id) {
            DB::transaction(function () use ($student_id) {
                $student = $this->studentService->findStudentById(intval($student_id));
                $this->studentSessionService->deleteStudentSessionById($student->studentSession->id);
                $this->userService->deleteUserById(intval($student->user_id));
                $this->studentService->deleteStudentById(intval($student->id));
            });
        }

        return $this->responseSuccess([], 'Students have been deleted successfully.');
    }

    public function downloadDemoFile()
    {
        $filePath = public_path('uploads/student_xlsx_file/student_demo.xlsx');

        if (file_exists($filePath)) {
            return response()->download($filePath, 'student_demo.xlsx', [
                'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            ]);
        }

        return response()->json([
            'status' => false,
            'message' => 'Demo file not found.',
        ], 404);
    }

    public function bulkImports(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'file' => 'required|file',
            'class_id' => 'required|integer|exists:classes,id',
            'academic_year' => 'required|integer|exists:academic_years,id',
            'section_id' => 'required|integer|exists:sections,id',
            'group' => 'required|string',
            'optional_subject' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return $this->responseError([], $validator->errors()->first(), 422);
        }

        $file = $request->file('file');
        $import = new StudentsImport();
        try {
            Excel::import($import, $file);
        } catch (Exception $e) {
            return $this->responseError([], 'Invalid Excel format or error during parsing.', 400);
        }

        $importedData = $import->data ?? [];
        $success = [];
        $failed = [];

        DB::beginTransaction();
        try {
            foreach ($importedData as $index => $row) {
                try {
                    $existingUser = User::where('phone', $row['phone'])->first();
                    if ($existingUser) {
                        $failed[] = [
                            'row' => $index + 1,
                            'error' => 'Duplicate phone number: ' . $row['phone'],
                            'data' => $row,
                        ];
                        continue;
                    }

                    $user = new User();
                    $user->institute_id = get_institute_id();
                    $user->branch_id = get_branch_id();

                    $user->name = $row['name'] ?? null;
                    $user->email = $row['phone'] ?? null;
                    $user->password = Hash::make($row['phone'] ?? 'password');
                    $user->user_type = 'Student';
                    $user->phone = $row['phone'] ?? null;
                    if (!$user->save()) {
                        throw new Exception("User creation failed.");
                    }

                    $student = new Student();
                    $student->institute_id = get_institute_id();
                    $student->branch_id = get_branch_id();
                    $student->user_id = $user->id;
                    $student->group = $request->group;
                    $student->first_name = $row['name'] ?? null;
                    $student->gender = $row['gender'] ?? null;
                    $student->religion = $row['religion'] ?? null;
                    $student->father_name = $row['fathers_name'] ?? null;
                    $student->mother_name = $row['mothers_name'] ?? null;
                    $student->phone = $row['phone'] ?? null;

                    if (!$student->save()) {
                        throw new Exception("Student creation failed.");
                    }

                    $studentSession = new StudentSession();
                    $studentSession->institute_id = get_institute_id();
                    $studentSession->branch_id = get_branch_id();
                    $studentSession->session_id = $request->academic_year;
                    $studentSession->student_id = $student->id;
                    $studentSession->class_id = $request->class_id;
                    $studentSession->section_id = $request->section_id;
                    $studentSession->roll = $row['roll_no'] ?? null;
                    $studentSession->optional_subject = $request->optional_subject ?? null;

                    if (!$studentSession->save()) {
                        throw new Exception("StudentSession creation failed.");
                    }

                    $success[] = [
                        'name' => $row['name'],
                        'phone' => $row['phone'],
                        'roll_no' => $row['roll_no'],
                    ];
                } catch (Exception $inner) {
                    $failed[] = [
                        'row' => $index + 1,
                        'error' => $inner->getMessage(),
                        'data' => $row,
                    ];
                }
            }

            DB::commit();

            return response()->json([
                'status' => true,
                'message' => 'Bulk upload completed',
                'successful_records' => $success,
                'failed_records' => $failed,
            ], 200);
        } catch (Exception $e) {
            DB::rollback();
            return $this->responseError([], 'Transaction failed: ' . $e->getMessage(), 500);
        }
    }
}
